######################################################################
######################################################################
# Function long.AirBase rearranges 2000-2010 AirBase data for the specified country, 
# components and measurement periods (read and stored by function read.AirBase or import.AirBase)  
# to long format.
#
# Input: 
# country (e.g. "DE" for germany), 
# vector of components names cp_name (e.g. c("o3","pm10")),
# vector of corresponding components codes cp_code (e.g. c("00007","00005")),
# vector of corresponding measurement periods cp_period (e.g. c("hour","day"))
#
# Output: 
# saves per component dataframes "country_period_component_long" 
# (e.g. DE_hour_o3_long, DE_day_pm10_long)
# to the corresponding country directory (e.g. DE) in ./Data/AirBase Data/ 
######################################################################
# author: M. Rehr
# LaMo: 2013-01-10
######################################################################
long.AirBase <- function(country, cp_name, cp_code, cp_period){
  
  cp_hour <- cp_code[which(cp_period=="hour")]
  cp_day <- cp_code[which(cp_period=="day")]
  
  names_hour <- cp_name[which(cp_period=="hour")]
  names_day <- cp_name[which(cp_period=="day")]
  
  # hourly data
  ##############
  for(i in 1:length(cp_hour)){
    
    # load data
    name <- paste(country, "hour", names_hour[i], sep="_")
    file <- paste("./Data/AirBase Data", country, name, sep="/")
    load(file)
    
    # data.frame long
    #################
    # data is of format: sn - date - value1 - ...- value24
    d <- data
    
    # rearrange data to long format
    Station <- rep(d$sn, each=24)
    
    # build Time variable in format "%Y-%m-%d %H:%M:%S" (Date & Hour)
    # assumption: times are given in GMT format, not in local formats. 
    days <- as.POSIXct(d$date, tz="GMT") - rep(3600, length(d$date)) 
    Time <- rep(days, each=24) + rep(seq(0, (23*3600), 3600), length(days))
    
    # stack values according to Time
    values <- t(d[,3:26]) 
    Value <- stack(as.data.frame(values))[,1]
    
    # data.frame of form: Station - Time - Value
    Data <- data.frame(Station, Time, Value)
    
    # NA coding (-999 -> NA, -888 -> NA) in values
    is.na(Data$Value) <- which(Data$Value==-999)
    is.na(Data$Value) <- which(Data$Value==-888)
    
    # store long
    long <- Data
    name_long <- paste(name, "long", sep="_")
    save(long, file=paste("./Data/AirBase Data", country, name_long, sep="/"))
    
  }
  
  
  # daily data: analogously
  #########################
  for(i in 1:length(cp_day)){
    
    name <- paste(country, "day", names_day[i], sep="_")
    file <- paste("./Data/AirBase Data", country, name, sep="/")
    load(file)
    
    # long
    #######
    # data is of format: sn - date - value1 - ...- value31
    d <- data
 
    # rearrange data to long format
    Station <- rep(unique(d$sn), each=4018)
    
    # build Time variable in format "%Y-%m-%d" (Date)   
    # ts from 2000-01-01 to 2010-12-31 (in days)
    date <- as.Date(0:4017, origin="2000-01-01")
    Time <- rep(date, times=length(unique(d$sn)))
    
    # stack values according to Time
    values <- t(d[,3:33]) 
    Value <- stack(as.data.frame(values))[,1]
        
    # discard not existing dates (like Feb. (29th,) 30th, 31st, April 31st etc.)
    year <- as.character(rep(2000:2010, each=12*31))
    month <- as.character(rep(rep(1:12, each=31)), each=11)
    month[1:(31*9)] <- paste("0", month[1:(31*9)], sep="")
    day <- as.character(1:31)
    day[1:9] <- paste("0", day[1:9], sep="")
    day <- rep(day,12*11)
    Time2 <- paste(year, month, day, sep="-")
    Time2 <- rep(Time2, length(unique(d$sn)))
    
    keep <- which(match(Time2, as.character(Time))!="NA")
    Value <- Value[keep]
    
    # data.frame of form: Station - Time - Value
    Data <- data.frame(Station, Time, Value)

    # NA coding 
    is.na(Data$Value) <- which(Data$Value==-999)
    is.na(Data$Value) <- which(Data$Value==-888)
    
    # store long
    long <- Data
    name_long <- paste(name, "long", sep="_")
    save(long, file=paste("./Data/AirBase Data", country, name_long, sep="/"))    
   
 }
  
}     
######################################################################  
######################################################################
