######################################################################
# author: M. Rehr
# LaMo: 2013-03-04
######################################################################

##############################################
##
## Data
##    load.DEM(country, cp_name)
##    load.allctrs.DEM(cp_name)
##    load.AirBase(country, cp_name, cp_period)
##    add.transforms(data)
##    deseasonalize(data, freq)
##
## Plots
##    make.ylab(cp_name)
##    make.xlab.DEM(cp_name)
##    make.xlab.AirBase(cp_period)
##    make.layout(n)
##
##############################################


######################################################################
# Data
######################################################################

###########################
# load DEM data for a given country, pollutant (and measurement period)
load.DEM <- function(country, cp_name){
  cp_period <- ifelse(cp_name=='pm10', 'day', 'hour')
  name <- paste(country, cp_period, cp_name, 'long', sep='_') 
  file <- paste('../Data/2010 DEM Data', country, name, sep='/')
  load(file)
  assign(paste(country, 'long', sep='_'), long)
}
###########################

########################### 
# load DEM data for a given pollutant (and measurement period) for all countries considered
# and return in one single data frame
load.allctrs.DEM <- function(cp_name){
  country <- list('ro', 'ch', 'cy')
  long <- lapply(country, load.DEM, cp_name=cp_name)
  names(long) <- c('ro', 'ch', 'cy')
  ro <- cbind(long$ro, 'ro') 
  ch <- cbind(long$ch, 'ch') 
  cy <- cbind(long$cy, 'cy') 
  names(ro) <- names(ch) <- names(cy) <- c(names(long$ro), 'Country')
  long <- rbind(ro,ch,cy)
  return(long)
}
###########################

########################### 
# load AirBase data for a given country, pollutant and measurement period
load.AirBase <- function(country, cp_name, cp_period){
  name <- paste(country, cp_period, cp_name, 'long', sep='_') 
  file <- paste('../Data/AirBase Data', country, name, sep='/')
  load(file)
  assign('data', long)
  assign(paste(country, 'long', sep='_'), data)
}
########################### 

########################### 
# add log10 and sqrt transforms to data
add.transforms <- function(data){
  log10Value <- log10(data$Value)
  sqrtValue <- sqrt(data$Value)
  return(cbind(data, log10Value, sqrtValue))
}
###########################

###########################
# apply STL decomposition and remove seasonal pattern
deseasonalize <- function(data, freq){
  ts_freq <- ts(data$Value, frequency=freq)
  ts_zoo <- as.zoo(ts_freq)
  ts <- na.approx(ts_zoo, method='linear')
  stl <- stl(ts, s.window=7) 
  trend <- stl$time.series[,2]
  remainder <- stl$time.series[,3]
  if (length(ts)==length(ts_zoo)){
    data$newValue <- as.numeric(trend+remainder)
  } else {
    data$newValue <- data$Value
    begin <- min(which(is.na(data$Value)==FALSE))
    end <- max(which(is.na(data$Value)==FALSE))
    data$newValue[begin:end] <- as.numeric(trend+remainder)
  }
  return(data)
}
###########################


######################################################################
# Plots
######################################################################

###########################
# pollutant and corresponding measurement unit as y axis in plots
make.ylab <- function(cp_name){
  if (cp_name=='pm10') {cp <- 'PM[10]'} else{
    if (cp_name=='o3') {cp <- 'O[3]'} else{
      if (cp_name=='co') {cp <- toupper(cp_name)} else{
        cp <- paste(toupper(substr(cp_name,1,2)),'[2]',sep="")
      }}}
  unit <- if (cp_name=='co') {expression(m*g/m^3)} else {expression(mu*g/m^3)}
  ylab <- noquote(paste(cp, ' ~ (',unit,')',sep=''))
  return(ylab)
}
###########################

###########################
# time and measurement unit as x axis in plots for DEM data (2010)
make.xlab.DEM <- function(cp_name){
  if (cp_name=='pm10') {period <- 'day'} else {period <- 'hour'}
  xlab <- paste('year 2010 ', expression('('), period, expression(')'), sep='')
  return(xlab)
}
###########################

###########################
# time and measurement unit as x axis in plots for AirBase data (2000-2010)
make.xlab.AirBase <- function(cp_period){
  xlab <- paste('years 2000 - 2010 ', expression('('), cp_period, expression(')'), sep='')
  return(xlab)
}
###########################

###########################
# layout for xyplots (number of columns, rows, pages)
make.layout <- function(n){
  c <- 0
  r <- n
  p <- 1
  if (n>60){p <- 2
            n <- ceiling(n/2)}
  c <- ceiling(n/10)
  r <- ceiling(n/c)
  l <- c(c, r, p) 
  return(l)
}
###########################

######################################################################
######################################################################