######################################################################
# author: M. Rehr
# LaMo: 2013-03-04
######################################################################

##############################################
##
## Thresholds and Indices
##    make.threshold(location, factor, dispersion)
##    get.idx.out(ts, threshold)
##
## Panel Functions
##    panel.fct.overview(x, y, q, text=T)
##    panel.fct.tukey(x, y, q, nd, text=T)
##    panel.fct.z(x, y, q, text=T)
##
## Captions
##    cap.out(cp_name, q, stat, transf)
##
##############################################


######################################################################
# Thresholds and Indices
######################################################################

###########################
# thresholds for outlier detection
make.threshold <- function(location, factor, dispersion){
  if(length(location)!=2) {location1 <- location2 <- location}
  else {location1 <- location[[1]]
        location2 <- location[[2]]}
  prod <- factor*dispersion
  list(location1 + prod, location2 - prod)
}
###########################

###########################
# indices of outliers
get.idx.out <- function(ts, threshold){
  which(ts > threshold[[1]] | ts < threshold[[2]])
}
###########################


######################################################################
# Panel Functions
######################################################################

###########################
# panel.function for static and q1 (overview heuristics)
panel.fct.overview <- function(x, y, q, text=TRUE) {
  # time series
  panel.xyplot(x, y, type="l")
  # median and quartiles
  if (q=="static"){
    quartiles <- quantile(y, probs=c(0.25,0.5,0.75), na.rm=TRUE)
    q1 <- quartiles[1]
    q2 <- quartiles[2]
    q3 <- quartiles[3]
  } else {
    #require(caTools)
    quartiles <- runquantile(y, k=2*q, probs=c(0.25,0.5,0.75)) 
    q1 <- quartiles[,1]
    q2 <- quartiles[,2]
    q3 <- quartiles[,3]
  }  
  # IQR (global)
  iqr <- IQR(y, na.rm=T)
  # thresholds
  b1 <- make.threshold(list(q3, q1), 1.5, iqr)
  llines(x, b1[[1]], col=grey(.8))
  llines(x, b1[[2]], col=grey(.8))
  b2 <- make.threshold(list(q3, q1), 3, iqr)
  llines(x, b2[[1]], col=grey(.5))
  llines(x, b2[[2]], col=grey(.5))
  # outliers
  out1 <- get.idx.out(y, b1)
  out2 <- get.idx.out(y, b2)
  panel.points(x[out1], y[out1], col=2, pch=20)
  panel.points(x[out2], y[out2], col=1, pch=20)
  if(text==TRUE){
    panel.text(x[out2], y[out2], gsub("2010-","",x[out2]), col=1, pos=3, offset=.5, cex=0.7)
  } 
  # mad_e (global)
  mad_e <- mad(y, na.rm=TRUE)
  b3 <- make.threshold(q2, 3, mad_e)
  llines(x, b3[[1]], col=grey(.6), lty=3)
  llines(x, b3[[2]], col=grey(.6), lty=3)
  b4 <- make.threshold(q2, 6, mad_e)
  llines(x, b4[[1]], col=grey(.3), lty=3)
  llines(x, b4[[2]], col=grey(.3), lty=3)
  # outliers
  out3 <- get.idx.out(y, b3)
  out4 <- get.idx.out(y, b4)
  panel.points(x[out3], y[out3], col=2, pch=1)
  panel.points(x[out4], y[out4], col=1, pch=1)
  if(text==TRUE) {
    panel.text(x[out4], y[out4], gsub("2010-","",x[out4]), col=2, pos=3, offset=.5, cex=0.7)
  }  
}
###########################

###########################
# panel.function for Tukey heuristic
panel.fct.tukey <- function(x, y, q, nd, text=TRUE){
  #loc
  quartiles <- runquantile(y, k=2*q, probs=c(0.75,0.25))
  q_u <- quartiles[,1]
  q_l <- quartiles[,2]
  loc <- list(q_u, q_l)  
  #disp
  iqr <- IQR(y, na.rm=TRUE)
  iqr_e <- iqr/1.349
  if(nd==TRUE) {disp <- iqr_e} else {disp <- iqr}  
  # quartiles
  #llines(x, q_u, col=grey(.8), lwd=0.8)
  #llines(x, q_l, col=grey(.8), lwd=0.8)
  # time series
  panel.xyplot(x, y, col=grey(.4), type="l", lwd=1.2)
  # thresholds
  b1 <- make.threshold(loc,1.5,disp)
  llines(x,b1[[1]],col=4)
  llines(x,b1[[2]],col=4)
  b2 <- make.threshold(loc,3,disp)
  llines(x,b2[[1]],col=3)
  llines(x,b2[[2]],col=3)
  # outliers
  out1 <-get.idx.out(y,b1)
  panel.points(x[out1],y[out1],col=2,pch=16)
  out2 <-get.idx.out(y,b2)
  panel.points(x[out2],y[out2],col=1,pch=16)
  if(text==TRUE) {
    panel.text(x[out2], y[out2], gsub("2010-","",x[out2]), col=2, pos=3, offset=.5)
  }
}
###########################

###########################
# panel.function for z heuristic
panel.fct.z <- function(x, y, q, text=TRUE) {
  #loc
  loc <- med <- runquantile(y, k=2*q, probs=c(0.5))  
  #disp
  disp <- mad_e <- mad(y, na.rm=TRUE)  
  # median 
  #llines(x, loc, col=grey(.8), lwd=0.8)
  # time series
  panel.xyplot(x, y, col=grey(.4), type="l", lwd=1.2)
  # thresholds  
  b1 <- make.threshold(loc,2,disp)
  llines(x,b1[[1]],col=grey(.7))
  llines(x,b1[[2]],col=grey(.7))
  b2 <- make.threshold(loc,3,disp)
  llines(x,b2[[1]],col=4)
  llines(x,b2[[2]],col=4)
  b3 <- make.threshold(loc,3.5,disp)
  llines(x,b3[[1]],col=grey(.7))
  llines(x,b3[[2]],col=grey(.7))
  b4 <- make.threshold(loc,6,disp)
  llines(x,b4[[1]],col=3)
  llines(x,b4[[2]],col=3)
  # outliers
  out1 <-get.idx.out(y,b1)
  panel.points(x[out1],y[out1],col=2, pch=20)
  out2 <-get.idx.out(y,b2)
  panel.points(x[out2],y[out2],col=2, pch=16)
  out3 <-get.idx.out(y,b3)
  panel.points(x[out3],y[out3],col=1)
  out4 <-get.idx.out(y,b4)
  panel.points(x[out4],y[out4],col=1, pch=16)
  if(text==TRUE) {
    panel.text(x[out4], y[out4], gsub("2010-","",x[out4]), col=2, pos=3, offset=.5)
  }  
}
###########################


######################################################################
# Captions
######################################################################

###########################
# automatic caption generation for plots
cap.out <- function(cp_name, q, stat, transf){
  if (stat=="all") {
    heuristic <- "overview heuristics,"
  } else {
    if (stat=="tukey") {
      heuristic <- "Tukey heuristic,"
    } else {
      heuristic <- "z heuristic,"
    }
  }
  if (transf=="o"){
    data <- "original data,"
  } else {
    data <- paste(transf, "transformed data,", sep=" ")
  }
  if (cp_name=="pm10") {
    static <- 365
    period <- "days"
  } else {
    static <- 365*24
    period <- "hours"
  }
  if (q=="static"){
    length <- static
  } else {
    length <- 2*q+1 
  }  
  caption <- paste(data, heuristic, "window length:", length, period, sep=" ")
  return(caption)
}
###########################

######################################################################
######################################################################