######################################################################
# author: M. Rehr
# LaMo: 2013-03-04
######################################################################

##############################################
##
## Peaks 
##    get.peaks(x,q)
##
## Filtering 
##    kz.f.sd(x, t, m, it)
##    kza.f.sd(kz, it.kza)
##
## Overview Method Functioning
##    plot.kza(kz, kza=NULL, q1=0.75, q2=0.95, tn=5, anticipated=NULL)
##
## Panel Functions
##    panel.fct.break(x, y, times, m, it.kz, it.kza, col)
##    panel.fct.break.layer(x, y, m, it.kz, it.kza, col, place)
##
##############################################


######################################################################
# Peaks
######################################################################

###########################
# find peaks (in variance) and return their indices (-> break time points)
get.peaks <- function(x,q){
  idx <- which(diff(sign(diff(x)))==-2)+1
  id <- which(x[idx]>=quantile(x, probs=q, na.rm=TRUE))
  p <- idx[id]
  return(p)
}
###########################


######################################################################
# Filtering
######################################################################

###########################
# run kz() and runsd()
kz.f.sd <- function(x, t, m, it){
  
  begin <- min(which(is.na(x)==FALSE))
  end <- max(which(is.na(x)==FALSE))
  
  Values <- x[begin:end]
  Time <- t[begin:end]
  
  Filter <- kz(x=Values, m=m, k=it)
  sd <- runsd(Filter, k=m)    
  
  kz <- list(Values, Time, Filter, sd, m, it)
  names(kz) <- c("Values", "Time", "Filter", "sd", "m", "it")
  return(kz)  
}
###########################

###########################
# run kza() and runsd()
kza.f.sd <- function(kz, it.kza){
  
  a <- kza(x=kz$Values, m=kz$m, k=it.kza, y=kz$Filter)
  Filter <- a$kza
  sd <- runsd(Filter, k=kz$m)    
  
  kza <- list(Filter, sd, it.kza)
  names(kza) <- c("Filter", "sd", "it.kza")
  return(kza)  
}
###########################


######################################################################
# Overview Method Functioning
######################################################################

###########################
# plot ts, kz/kza, var, breaks
plot.kza <- function(kz, kza=NULL, q1=0.75, q2=0.95, tn=5, anticipated=NULL){
  
  Time <- kz$Time
  Values <- kz$Values
  
  if(length(kza)==0){
    Filter <- kz$Filter
    Variance <- (kz$sd)^2
    main <- "kz filter"
  }
  else{
    Filter <- kza$Filter
    Variance <- (kza$sd)^2
    main <- "kza filter"
  }
  
  p1 <- get.peaks(Variance, q1)
  p2 <- get.peaks(Variance, q2)
  
  # timeseries
  plot(Time, Values, type="l", main=main)
  lines(Time, Filter, col=2)
    # add anticipated breaks
    points(x=anticipated, y=rep(min(Values),length(anticipated)), pch=17, col=4, cex=1.3)
    abline(v=anticipated, col=4, lty=3)
    #text(x=anticipated, y=max(Values)-max(Values)/10, labels=substr(anticipated,6,10), col=4)
  
  # smoothed 
  plot(Time, Filter, type="l", ylim=c(0, 1.5*max(kz$Filter, na.rm=T)))
    abline(v=anticipated, col=4, lty=3)
  
  # variance (smoothed)
  plot(Time, Variance, type="l", ylim=1.5*range(Variance, na.rm=T))
  points(Time[p1], Variance[p1], col=grey(.5))
  points(Time[p2], Variance[p2], col=2, pch=16)
  if(length(p2)>tn)
  {idx <- p2[rev(order(Variance[p2]))[1:tn]]}
  else {idx <- p2}
  text(Time[idx], Variance[idx], substr(Time[idx],6,10), col=2, pos=3, offset=.5)
    abline(v=anticipated, col=4, lty=3)
  
  # timeseries + kza breaks
  plot(Time, Values, type="l")
  #points(x=Time[p1], y=rep(min(Values),length(p1)), pch=17, col=grey(.5), cex=1.3)
  #points(x=Time[p2], y=rep(min(Values),length(p2)), pch=17, col=2, cex=1.3)
  abline(v=Time[p1], col=grey(.5), lty=3)
  abline(v=Time[p2], col=2)
    abline(v=anticipated, col=4, lty=3)
}
###########################


######################################################################
# Panel Functions
######################################################################

###########################
# panel.function for break detection
panel.fct.break <- function(x, y, times, m, it.kz, it.kza, col){
  # NAs
  panel.abline(v=x[which(is.na(y))], col=grey(.85), lwd=0.5)
  pn <- packet.number()
  # runs of equal values
  panel.abline(v=unlist(times[pn]), col=grey(.7), lwd=0.5, lty=1)
  # compute kz and kza
  kz <- kz.f.sd(x=y, t=x, m=m, it=it.kz)
  kza <- kza.f.sd(kz=kz, it.kza=it.kza)
  # compute variance and get peaks
  Variance <- (kza$sd)^2
  b <- c(0.75, 0.95, 0.975, 0.995) 
  p <- sapply(b, get.peaks, x=Variance)
  # time series
  panel.xyplot(x, y, type="l", col=grey(.8))
  # detected breaks per threshold
  panel.points(x=kz$Time[p[[2]]], y=kza$Filter[p[[2]]], pch=25, col=col, fill='yellow', cex=1)
  panel.points(x=kz$Time[p[[3]]], y=kza$Filter[p[[3]]], pch=25, col=col, fill='orange', cex=1)
  panel.points(x=kz$Time[p[[4]]], y=kza$Filter[p[[4]]], pch=25, col=col, fill=2, cex=1)
  # kza
  panel.lines(kz$Time, kza$Filter, col=col)
  # suspected change points
  if (length(changes)==length(unique(data$Station))){
    panel.points(x=unlist(changes[pn]), y=0, pch=17, col=1, cex=0.9)
  }
}
###########################

###########################
# panel.function (layer) for break detection
panel.fct.break.layer <- function(x, y, m, it.kz, it.kza, col, place){
  # compute kz and kza
  kz <- kz.f.sd(x=y, t=x, m=m, it=it.kz)
  kza <- kza.f.sd(kz=kz, it.kza=it.kza)
  # compute variance and get peaks
  Variance <- (kza$sd)^2
  b <- c(0.75, 0.95, 0.975, 0.995) 
  p <- sapply(b, get.peaks, x=Variance)
  # detected breaks per threshold
  panel.points(x=kz$Time[p[[2]]], y=kza$Filter[p[[2]]]+place, pch=25, col=col, fill='yellow', cex=1)
  panel.points(x=kz$Time[p[[3]]], y=kza$Filter[p[[3]]]+place, pch=25, col=col, fill='orange', cex=1)
  panel.points(x=kz$Time[p[[4]]], y=kza$Filter[p[[4]]]+place, pch=25, col=col, fill=2, cex=1) 
  # kza
  panel.lines(kz$Time, (kza$Filter+place), col=col)
}
###########################

######################################################################
######################################################################